import UIKit
import SnapKit
import DMScrollBar

final class ViewController: UIViewController, DMScrollBarDelegate {

    private let scrollView = UIScrollView()
    private let stackView = UIStackView()
    private let doubleTapGR = UITapGestureRecognizer()
    private let imagesArray: [UIImage] = [
        .init(named: "1") ?? .init(),
        .init(named: "2") ?? .init(),
        .init(named: "3") ?? .init(),
        .init(named: "4") ?? .init()
    ]

    override func viewDidLoad() {
        super.viewDidLoad()

        setupView()
    }

    func indicatorTitle(forOffset offset: CGFloat) -> String? {
        nil
    }
}

// MARK: - Private setup view
private extension ViewController {
    func setupHierarchy() {
        view.addSubview(scrollView)
        scrollView.addSubview(stackView)
    }
    
    func setupConstraints() {
        scrollView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        stackView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
            make.width.equalToSuperview()
        }
    }
    
    func configureViews() {
        scrollView.delegate = self
        
        scrollView.minimumZoomScale = 1.0
        scrollView.maximumZoomScale = 4.0
        scrollView.showsVerticalScrollIndicator = false
        scrollView.showsHorizontalScrollIndicator = false
        
        stackView.axis = .vertical
        stackView.spacing = 16.0
        
        doubleTapGR.addTarget(self, action: #selector(doubleTapAction))
        doubleTapGR.numberOfTapsRequired = 2
        view.addGestureRecognizer(doubleTapGR)
    }
    
    func configureCustomScrollView() {
        let configuration = DMScrollBar.Configuration(
            isAlwaysVisible: false,
            hideTimeInterval: 0.4,
            shouldDecelerate: true,
            indicator: .init(
                normalState: .init(
                    backgroundColor: .clear,
                    image: .init(named: "scroll"),
                    imageSize: .init(width: 24.0, height: 51.0)
                ),
                activeState: .custom(
                    config: .init(
                        backgroundColor: .clear,
                        image: .init(named: "scroll"),
                        imageSize: .init(width: 24.0, height: 51.0)
                    )
                ),
                insetsFollowsSafeArea: true
            )
        )
        scrollView.configureScrollBar(with: configuration, delegate: self)
    }
    
    @objc func doubleTapAction(_ gestureRecognizer: UITapGestureRecognizer) {
        // If the zoom scale is greater than the minimum zoom scale, set it to the minimum zoom scale
        if scrollView.zoomScale > scrollView.minimumZoomScale {
            scrollView.setZoomScale(scrollView.minimumZoomScale, animated: true)
        }
    }

    func constructViews() {
        for image in imagesArray {
            let imageView = UIImageView()
            imageView.contentMode = .scaleAspectFit
            imageView.image = image
            stackView.addArrangedSubview(imageView)
            imageView.snp.makeConstraints { make in
                make.width.equalToSuperview()
                make.height
                    .equalTo(stackView.snp.width)
                    .multipliedBy(1.0 / 1.0)
            }
        }
    }
    
    func setupView() {
        setupHierarchy()
        setupConstraints()
        constructViews()
        configureViews()
        configureCustomScrollView()
    }
}

// MARK: - UIScrollViewDelegate
extension ViewController: UIScrollViewDelegate {
    func viewForZooming(in scrollView: UIScrollView) -> UIView? {
        // Return the scroll view's first subview, which is the first page
        return scrollView.subviews.first
    }

    func scrollViewDidZoom(_ scrollView: UIScrollView) {
        if scrollView.zoomScale < scrollView.minimumZoomScale {
            scrollView.zoomScale = scrollView.minimumZoomScale
        }
    }
}
